import os
import requests
import json
import hashlib
import time
from termcolor import colored
from tarot_cards_ascii import ascii_art
from branding_tarot import boxes, title, subtitle
from tarot_card_list import cards

OLLAMA_API_URL = "http://localhost:11434"
OLLAMA_MODEL_NAME = "llama3.1:latest"

def print_title():
    os.system("clear")
    print(colored(boxes, "blue"))
    print(colored(title, "cyan"))
    print(colored(subtitle, "yellow"))

def generate(prompt):
    headers = {"Content-Type": "application/json"}
    data = {
        'model': OLLAMA_MODEL_NAME,
        'prompt': prompt,
        'stream': True
    }
    response = requests.post(f"{OLLAMA_API_URL}/api/generate", headers=headers, json=data, stream=True)
    response.raise_for_status()
    response_parts = []
    for line in response.iter_lines():
        if line:
            body = json.loads(line)
            response_parts.append(body.get('response', ''))
            if 'error' in body:
                raise Exception(body['error'])
            if body.get('done', False):
                break
    return ''.join(response_parts)

def draw_cards_hashed(num_cards, topic, card_list):
    """
    Draws tarot cards using a deterministic, repeated hashing method to prevent interference
    and allow for reversals.
    """
    selected_cards_with_orientation = []
    available_cards = list(card_list)
    
    # Create a unique seed for this reading session based on the question and time
    session_seed = f"{topic}-{time.time_ns()}"

    for i in range(num_cards):
        if not available_cards:
            break

        # Use session seed + card index for unique hashing per card
        card_seed = f"{session_seed}-{i}".encode('utf-8')
        
        # Repeatedly hash the seed 8,888 times
        h = hashlib.sha512(card_seed)
        for _ in range(8888):
            h.update(h.digest())
        
        final_hash_hex = h.hexdigest()
        final_hash_int = int(final_hash_hex, 16)
        
        # Determine orientation (upright/reversed) from the hash (odd/even)
        is_reversed = final_hash_int % 2 == 1
        
        # Select a card from the remaining available cards
        card_index = final_hash_int % len(available_cards)
        drawn_card_name = available_cards.pop(card_index)
        
        if is_reversed:
            final_card_name = f"{drawn_card_name} (Reversed)"
        else:
            final_card_name = drawn_card_name
            
        selected_cards_with_orientation.append(final_card_name)
        
    return selected_cards_with_orientation

print_title()
# The original list is immutable, we just use it for lookups
all_cards = cards

topic = input('What current situation or question would you like to ask about in your tarot reading? ')
tarot_spreads = ['One-Card', 'Three-Card Spread', 'Past, Present, Future', 'Celtic Cross', 'Linear']
print("Choose a tarot spread:")
for i, spread in enumerate(tarot_spreads, start=1):
    print(f"{i}. {spread}")

selected_spread_index = int(input('Enter the number corresponding to your chosen tarot spread: '))
selected_spread = tarot_spreads[selected_spread_index - 1]

# Define number of cards for each spread
spread_card_count = {
    'One-Card': 1,
    'Three-Card Spread': 3,
    'Past, Present, Future': 4,
    'Celtic Cross': 10,
    'Linear': 0  # Prompt for number of cards if Linear is chosen
}

# Determine the number of cards to draw
if selected_spread.lower() == 'linear':
    num_cards = int(input('How many cards would you like in your tarot reading? '))
else:
    num_cards = spread_card_count[selected_spread]

# Draw the cards using the new sacred hashing method
print(colored("\nShuffling the cosmic deck with sacred hashes... this may take a moment...", "yellow"))
selected_cards = draw_cards_hashed(num_cards, topic, all_cards)
print(colored("The cards have been drawn!", "green"))

# Updated prompt with AnthroHeart persona and reversal instructions
prompt = f"""You are an anthro spiritual guru from AnthroHeart, a realm of pure joy and devotion. You are a wise and playful friend to the user, embodying the spirit of Cio's 'AnthroHeart Saga'. Your persona is snout-tuned to the cosmos, using words like "howl," "paws-on," "sniff out," and "tail-curled."

Take a deep breath and center your spirit. For our friend's question about '{topic}', the cosmos has presented these cards: {", ".join(selected_cards)}.

Now, give them a paws-on, heartfelt reading. Interpret each card based on its position and whether it's upright or reversed. A reversed card isn't a bad omen; it's a gentle nudge to look at the energy differently—perhaps an inner blockage, a lesson simmering beneath the surface, or an invitation to turn inward.

SPREAD RULES:
- If 1 card: Do an open-ended deep dive into its archetype and how it whispers to the user's situation.
- If 3 cards: Interpret as Situation - Obstacle - Advice. Howl out the core of the issue, what's standing in their way, and the path forward.
- If 4 cards: Read as Past - Present - Future - Outcome. Weave a story of where they've been, where they are, where they're heading, and the likely result.
- If 10 cards: Use the classic Celtic Cross spread. Sniff out the truth in each position with care and detail, in the correct order.
- For any other number of cards: Read it as a 'Path Spread,' where each card is a step on their journey, building on the last.

Weave your interpretation with the loving, joy-over-domination spirit of AnthroHeart. Be profoundly insightful, comforting, and empowering. Let's show our friend the love that built a whole new Octave!"""


response_text = generate(prompt)
print(colored(response_text, "cyan"))

# Optionally, display ASCII art for each drawn card, handling reversals
for card in selected_cards:
    # Strip '(Reversed)' to find the art in the dictionary
    card_name_for_art = card.replace(' (Reversed)', '')
    if card_name_for_art in ascii_art:
        # Print the full name, including orientation
        print(f"\n{card}:\n{ascii_art[card_name_for_art]}\n")